/**
 * Ask People Extension - Website Content Script
 * Injected into the Ask People website to detect queries and communicate with extension
 */

(function() {
  'use strict';

  // Use browser API with chrome fallback for cross-browser compatibility
  const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

  // Detect the API URL from the current page
  const apiUrl = window.location.origin;

  // Start a query session helper
  function startQuerySession(query, source = null, intent = null) {
    console.log('[Extension] Starting query session:', query, 'source:', source, 'intent:', intent);

    browserAPI.runtime.sendMessage({
      type: 'START_QUERY_SESSION',
      query: query,
      apiUrl: apiUrl,
      source: source,
      intent: intent
    }).then(response => {
      console.log('[Extension] Session started:', response);
    }).catch(err => {
      console.error('[Extension] Failed to start session:', err);
    });
  }

  // Listen for "no YAGO results" event (Fact intent)
  window.addEventListener('askpeople-no-yago-results', (event) => {
    const { query } = event.detail;
    console.log('[Extension] Detected query with no YAGO results:', query);
    startQuerySession(query, 'yago', 'fact');
  });

  // Listen for "external search" event (How-to, Opinion, etc.)
  window.addEventListener('askpeople-external-search', (event) => {
    const { query, source, intent } = event.detail;
    console.log('[Extension] Detected external search:', query, 'via', source);
    startQuerySession(query, source, intent);
  });

  // Also listen for message-based approach (alternative)
  window.addEventListener('message', (event) => {
    if (event.source !== window) return;

    if (event.data.type === 'ASKPEOPLE_NO_YAGO_RESULTS') {
      const { query } = event.data;
      console.log('[Extension] Detected query via postMessage (no YAGO):', query);
      startQuerySession(query, 'yago', 'fact');
    }

    if (event.data.type === 'ASKPEOPLE_EXTERNAL_SEARCH') {
      const { query, source, intent } = event.data;
      console.log('[Extension] Detected external search via postMessage:', query);
      startQuerySession(query, source, intent);
    }
  });

  // Inject a small indicator that the extension is active
  const indicator = document.createElement('div');
  indicator.id = 'askpeople-extension-indicator';
  indicator.style.cssText = 'display: none;';
  indicator.dataset.extensionActive = 'true';
  document.body.appendChild(indicator);

  console.log('[Ask People Extension] Content script loaded on website');
})();

