/**
 * Ask People Extension - Text Selection Content Script
 * Detects highlighted text and shows option to save as answer
 * Also shows floating session bar when there's an active search session
 */

(function() {
  'use strict';

  // Use browser API with chrome fallback for cross-browser compatibility
  const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

  let saveButton = null;
  let selectedText = '';
  let sessionBar = null;

  // Create the floating save button
  function createSaveButton() {
    const btn = document.createElement('div');
    btn.id = 'askpeople-save-btn';
    btn.innerHTML = '💾 Save as answer';
    btn.style.cssText = `
      position: absolute;
      background: #2563eb;
      color: white;
      padding: 8px 12px;
      border-radius: 6px;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 13px;
      cursor: pointer;
      z-index: 999999;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      display: none;
      user-select: none;
      transition: background 0.2s;
    `;
    btn.addEventListener('mouseenter', () => {
      btn.style.background = '#1d4ed8';
    });
    btn.addEventListener('mouseleave', () => {
      btn.style.background = '#2563eb';
    });
    btn.addEventListener('click', handleSaveClick);
    document.body.appendChild(btn);
    return btn;
  }

  // Handle save button click
  async function handleSaveClick(e) {
    e.preventDefault();
    e.stopPropagation();

    if (!selectedText) return;

    // Check if there's an active session
    try {
      const response = await browserAPI.runtime.sendMessage({ type: 'GET_SESSION' });

      if (!response.session || !response.session.query) {
        showFeedback('No active search session', 'error');
        return;
      }

      // Try to get context (e.g., permalink for Reddit/SO comments)
      const context = getSelectionContext();

      // Save the selected text
      const saveResponse = await browserAPI.runtime.sendMessage({
        type: 'SAVE_TEXT_AS_ANSWER',
        text: selectedText,
        url: window.location.href,
        context: context
      });

      if (saveResponse.success) {
        showFeedback('Answer saved!', 'success');
        // Don't hide - showFeedback will handle hiding after delay
      } else {
        showFeedback('Failed to save', 'error');
      }
    } catch (error) {
      console.error('[Extension] Save error:', error);
      showFeedback('Error saving', 'error');
    }
    // Note: Don't call hideSaveButton() here - showFeedback handles it
  }

  // Try to get context for the selection (e.g., comment permalink)
  function getSelectionContext() {
    const selection = window.getSelection();
    if (!selection.rangeCount) return null;

    const range = selection.getRangeAt(0);
    let element = range.commonAncestorContainer;
    
    // Walk up to find a container with useful context
    while (element && element !== document.body) {
      if (element.nodeType === Node.ELEMENT_NODE) {
        // Look for permalink or timestamp links (common on Reddit, SO, etc.)
        const permalink = element.querySelector('a[href*="comment"], a[href*="permalink"], a.timestamp');
        if (permalink) {
          return `Permalink: ${permalink.href}`;
        }
        
        // Check for data attributes that might contain IDs
        if (element.dataset.commentId || element.dataset.id) {
          return `Comment ID: ${element.dataset.commentId || element.dataset.id}`;
        }
      }
      element = element.parentNode;
    }

    return null;
  }

  // Show feedback message
  function showFeedback(message, type) {
    if (!saveButton) return;

    if (type === 'success') {
      saveButton.innerHTML = '✓ Success';
      saveButton.style.background = '#10b981 !important;';
      saveButton.style.color = 'white';
      saveButton.style.fontWeight = '600';

      // Keep showing success for 3 seconds then hide
      setTimeout(() => {
        hideSaveButton();
        // Reset styles for next use
        saveButton.innerHTML = '💾 Save as answer';
        saveButton.style.background = '#2563eb';
        saveButton.style.color = 'white';
        saveButton.style.fontWeight = 'normal';
      }, 2000);
    } else {
      saveButton.innerHTML = `✗ ${message}`;
      saveButton.style.background = '#ef4444';

      setTimeout(() => {
        saveButton.innerHTML = '💾 Save as answer';
        saveButton.style.background = '#2563eb';
      }, 2000);
    }
  }

  // Show save button near selection
  function showSaveButton(x, y) {
    if (!saveButton) {
      saveButton = createSaveButton();
    }
    saveButton.style.left = `${x}px`;
    saveButton.style.top = `${y + window.scrollY + 10}px`;
    saveButton.style.display = 'block';
  }

  // Hide save button
  function hideSaveButton() {
    if (saveButton) {
      saveButton.style.display = 'none';
    }
  }

  // Handle text selection
  document.addEventListener('mouseup', async (e) => {
    // Ignore clicks on the save button itself
    if (e.target.id === 'askpeople-save-btn') return;

    const selection = window.getSelection();
    const text = selection.toString().trim();

    if (text.length > 10) { // Minimum 10 chars to show button
      selectedText = text;
      
      // Check if there's an active session before showing button
      try {
        const response = await browserAPI.runtime.sendMessage({ type: 'GET_SESSION' });
        if (response.session && response.session.query) {
          const rect = selection.getRangeAt(0).getBoundingClientRect();
          showSaveButton(rect.left + rect.width / 2 - 60, rect.bottom);
        }
      } catch (err) {
        // Extension context may be invalid, ignore
      }
    } else {
      selectedText = '';
      hideSaveButton();
    }
  });

  // Hide button when clicking elsewhere
  document.addEventListener('mousedown', (e) => {
    if (e.target.id !== 'askpeople-save-btn') {
      hideSaveButton();
    }
  });

  // ========== FLOATING SESSION PANEL ==========
  // Shows in top-right corner when there's an active search session
  // Styled like the extension popup

  let sessionBarFeedback = null;

  function createSessionBar(query) {
    if (sessionBar) {
      sessionBar.remove();
    }

    const bar = document.createElement('div');
    bar.id = 'askpeople-session-bar';
    bar.innerHTML = `
      <div class="askpeople-panel-header">
        <span class="askpeople-panel-close" id="askpeople-panel-close">✕</span>
      </div>
      <div class="askpeople-panel-query">${escapeHtml(query)}</div>
      <button class="askpeople-panel-btn askpeople-panel-save" id="askpeople-panel-save">
        💾 Save this page as answer
      </button>
            <div class="askpeople-panel-feedback" id="askpeople-panel-feedback"></div>

      <div class="askpeople-panel-hint">
        Tip: Highlight text on any page and right-click to save it as an answer.
      </div>
      <button class="askpeople-panel-btn askpeople-panel-done" id="askpeople-panel-done">
        ✓ Done searching
      </button>
    `;

    // Add styles
    const style = document.createElement('style');
    style.id = 'askpeople-session-bar-styles';
    style.textContent = `
      #askpeople-session-bar {
        position: fixed;
        top: 16px;
        right: 16px;
        width: 260px;
        background: #fff;
        border-radius: 12px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.15);
        z-index: 999999;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        font-size: 14px;
        color: #333;
        padding: 16px;
      }
      #askpeople-session-bar .askpeople-panel-header {
        display: flex;
        justify-content: flex-end;
        margin-bottom: 8px;
        position: absolute;
  top: 2%;
  right: 5%;
      }
      #askpeople-session-bar .askpeople-panel-close {
        cursor: pointer;
        color: #999;
        font-size: 16px;
        padding: 4px;
      }
      #askpeople-session-bar .askpeople-panel-close:hover { color: #333; }
      #askpeople-session-bar .askpeople-panel-query {
        font-weight: 500;
        margin-bottom: 12px;
        word-break: break-word;
        line-height: 1.4;
      }
      #askpeople-session-bar .askpeople-panel-feedback {
        padding: 10px;
        border-radius: 8px;
        margin-bottom: 12px;
        text-align: center;
        display: none;
      }
      #askpeople-session-bar .askpeople-panel-feedback.success {
        display: block;
        background: #d1fae5;
        color: #065f46;
      }
      #askpeople-session-bar .askpeople-panel-feedback.error {
        display: block;
        background: #fee2e2;
        color: #991b1b;
      }
      #askpeople-session-bar .askpeople-panel-btn {
        display: block;
        width: 100%;
        // padding: 12px 16px;
        border: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 400;
        cursor: pointer;
        margin-bottom: 8px;
        transition: background 0.2s;
        text-align: center;
        height: 42px;
      }
      #askpeople-session-bar .askpeople-panel-save {
        background: rgb(207, 231, 207);
        color: black;
      }
      #askpeople-session-bar .askpeople-panel-save:hover { background: rgb(192, 215, 192); }
      #askpeople-session-bar .askpeople-panel-save:disabled {
        opacity: 0.7;
        cursor: not-allowed;
      }
      #askpeople-session-bar .askpeople-panel-done {
        background: #f3f4f6;
        color: #666;
        font-size: 13px;
        // padding: 10px 16px;
        margin-top: 8px;
        margin-bottom: 0;
        height: 41px;
      }
      #askpeople-session-bar .askpeople-panel-done:hover { background: #e5e7eb; }
      #askpeople-session-bar .askpeople-panel-hint {
        font-size: 13px;
        color: #666;
        text-align: center;
        line-height: 1.4;
        margin-top: 4px;
      }
    `;

    // Remove old styles if they exist
    const oldStyle = document.getElementById('askpeople-session-bar-styles');
    if (oldStyle) oldStyle.remove();

    document.head.appendChild(style);
    document.body.appendChild(bar);

    // Event listeners
    document.getElementById('askpeople-panel-save').addEventListener('click', async () => {
      const btn = document.getElementById('askpeople-panel-save');
      const feedback = document.getElementById('askpeople-panel-feedback');
      const originalText = btn.textContent;
      btn.textContent = 'Saving...';
      btn.disabled = true;

      try {
        const response = await browserAPI.runtime.sendMessage({
          type: 'SAVE_PAGE_AS_ANSWER',
          url: window.location.href,
          title: document.title
        });

        if (response.success) {
          feedback.textContent = '✓ Answer saved!';
          feedback.className = 'askpeople-panel-feedback success';
          btn.textContent = originalText;
          btn.disabled = false;
        } else {
          feedback.textContent = '✗ Failed to save. Please try again.';
          feedback.className = 'askpeople-panel-feedback error';
          btn.textContent = originalText;
          btn.disabled = false;
        }
      } catch (err) {
        feedback.textContent = '✗ Error saving. Please try again.';
        feedback.className = 'askpeople-panel-feedback error';
        btn.textContent = originalText;
        btn.disabled = false;
      }
    });

    document.getElementById('askpeople-panel-done').addEventListener('click', async () => {
      await browserAPI.runtime.sendMessage({ type: 'END_SESSION' });
      hideSessionBar();
    });

    document.getElementById('askpeople-panel-close').addEventListener('click', () => {
      hideSessionBar();
    });

    sessionBar = bar;
  }

  function hideSessionBar() {
    if (sessionBar) {
      sessionBar.remove();
      sessionBar = null;
    }
    const style = document.getElementById('askpeople-session-bar-styles');
    if (style) style.remove();
  }

  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // Check if we're on the Ask People site itself
  function isAskPeopleSite() {
    const hostname = window.location.hostname;
    return hostname.includes('localhost') ||
           hostname.includes('127.0.0.1') ||
           hostname.includes('fly.dev') ||
           hostname.includes('askpeople.co');
  }

  // Check for active session on page load and show bar
  async function checkAndShowSessionBar() {
    try {
      const response = await browserAPI.runtime.sendMessage({ type: 'GET_SESSION' });
      if (response.session && response.session.query) {
        // Don't show on Ask People site itself
        if (!isAskPeopleSite()) {
          createSessionBar(response.session.query);
        }
      }
    } catch (err) {
      // Extension context may be invalid
    }
  }

  // Check on page load
  checkAndShowSessionBar();

  // Listen for session changes
  browserAPI.runtime.onMessage.addListener((message) => {
    if (message.type === 'SESSION_STARTED') {
      // Don't show on Ask People site itself
      if (!isAskPeopleSite()) {
        createSessionBar(message.query);
      }
    } else if (message.type === 'SESSION_ENDED') {
      hideSessionBar();
    }
  });

  console.log('[Ask People Extension] Selection script loaded');
})();

